/*____________________________________________________________________________
		Copyright (C) 2000 Network Associates, Inc.
        All rights reserved.

        $Id: CUnicodeString.cpp,v 1.6 1999/10/11 07:06:04 nryan Exp $
____________________________________________________________________________*/

#include "pgpClassesConfig.h"
#include "CString.h"
#include "CUnicodeString.h"

_USING_PGP

// Class CUnicodeString member functions

CUnicodeString::CUnicodeString() : mWCharArray(kDefaultUStringSize)
{
	Init();
	Status() = mWCharArray.Status();
}

CUnicodeString::CUnicodeString(const CUnicodeString& custr) : 
	mWCharArray(kDefaultUStringSize)
{
	Init();

	if (Status().IsntError())
		Status() = Assign(static_cast<const UNICODE_STRING&>(custr));
}

CUnicodeString::CUnicodeString(const UNICODE_STRING& ustr) : 
	mWCharArray(kDefaultUStringSize)
{
	Init();

	if (Status().IsntError())
		Status() = Assign(ustr);
}

CUnicodeString::CUnicodeString(const WCHAR *wstr) : 
	mWCharArray(kDefaultUStringSize)
{
	pgpAssertAddrValid(wstr, WCHAR);

	Init();

	if (Status().IsntError())
		Status() = Assign(wstr);
}

CUnicodeString::CUnicodeString(const char *str) : 
	mWCharArray(kDefaultUStringSize)
{
	pgpAssertStrValid(str);

	Init();

	if (Status().IsntError())
		Status() = Assign(str);
}

CUnicodeString::CUnicodeString(char c) : 
	mWCharArray(kDefaultUStringSize)
{
	Init();

	if (Status().IsntError())
		Status() = Assign(c);
}

CUnicodeString& 
CUnicodeString::operator=(const CUnicodeString& custr)
{
	Status() = Assign(static_cast<const UNICODE_STRING&>(custr));

	return *this;
}

CUnicodeString& 
CUnicodeString::operator=(const UNICODE_STRING& ustr)
{
	Status() = Assign(ustr);

	return *this;
}

CUnicodeString& 
CUnicodeString::operator=(const WCHAR *wstr)
{
	pgpAssertAddrValid(wstr, WCHAR);
	Status() = Assign(wstr);

	return *this;
}

CUnicodeString& 
CUnicodeString::operator=(const char *str)
{
	pgpAssertStrValid(str);
	Status() = Assign(str);

	return *this;
}

CUnicodeString& 
CUnicodeString::operator=(char c)
{
	Status() = Assign(c);
	return *this;
}

CUnicodeString& 
CUnicodeString::operator+=(const UNICODE_STRING& ustr)
{
	Status() = Append(ustr);
	return *this;
}

CUnicodeString& 
CUnicodeString::operator+=(const WCHAR *wstr)
{
	pgpAssertAddrValid(wstr, WCHAR);
	Status() = Append(wstr);

	return *this;
}

CUnicodeString& 
CUnicodeString::operator+=(const char *str)
{
	pgpAssertStrValid(str);
	Status() = Append(str);

	return *this;
}

CUnicodeString& 
CUnicodeString::operator+=(char c)
{
	Status() = Append(c);
	return *this;
}

PGPBoolean 
CUnicodeString::IsEmpty() const
{
	return ((Length() == 0) || (GetAt(0) == '\0'));
}

PGPBoolean 
CUnicodeString::Compare(const UNICODE_STRING& ustr) const
{
	return (RtlCompareUnicodeString(const_cast<PUNICODE_STRING>(&ustr), 
		const_cast<PUNICODE_STRING>(&mUniData), FALSE) == 0);
}

PGPBoolean 
CUnicodeString::Compare(const WCHAR *wstr) const
{
	UNICODE_STRING	tempUstr;

	pgpAssertAddrValid(wstr, WCHAR);
	RtlInitUnicodeString(&tempUstr, wstr);

	return Compare(tempUstr);
}

PGPBoolean 
CUnicodeString::CompareNoCase(const UNICODE_STRING& ustr) const
{
	return (RtlCompareUnicodeString(const_cast<PUNICODE_STRING>(&ustr), 
		const_cast<PUNICODE_STRING>(&mUniData), TRUE) == 0);
}

PGPBoolean 
CUnicodeString::CompareNoCase(const WCHAR *wstr) const
{
	pgpAssertAddrValid(wstr, WCHAR);

	UNICODE_STRING	tempUstr;
	RtlInitUnicodeString(&tempUstr, wstr);

	return CompareNoCase(tempUstr);
}

CComboError 
CUnicodeString::ToAscii(CString& cstr) const
{
	ANSI_STRING	ansiString;
	CComboError	error;

	error = cstr.GetBuffer(Length() + 1, ansiString.Buffer);

	if (error.IsntError())
	{
		ansiString.Length = static_cast<PGPUInt16>(Length());
		ansiString.MaximumLength = Length() + 1;

		error.err = RtlUnicodeStringToAnsiString(&ansiString, 
			const_cast<PUNICODE_STRING>(&mUniData), FALSE);

		if (error.HaveNonPGPError())
			error.pgpErr = kPGPError_StringOpFailed;

		cstr.ReleaseBuffer();
	}

	return error;
}

CComboError 
CUnicodeString::ZeroTerminate()
{
	CComboError	error;

	if (LastWChar() != '\0')
		error = Append("\0");

	return error;
}

void 
CUnicodeString::RemoveZeroTermination()
{
	if (LastWChar() == '\0')
		mUniData.Length -= sizeof(WCHAR);
}

void 
CUnicodeString::Empty()
{
	mUniData.Length = 0;
}

CComboError 
CUnicodeString::Assign(const UNICODE_STRING& ustr)
{
	CComboError	error;

	if (mWCharArray.Get() == ustr.Buffer)
		return error;

	PGPUInt32	newLength	= ustr.Length;
	error = Resize(newLength/sizeof(WCHAR));

	if (error.IsntError())
		RtlCopyUnicodeString(&mUniData, const_cast<PUNICODE_STRING>(&ustr));

	return error;
}

CComboError 
CUnicodeString::Assign(const WCHAR *wstr)
{
	UNICODE_STRING	tempUstr;

	pgpAssertAddrValid(wstr, WCHAR);
	RtlInitUnicodeString(&tempUstr, wstr);

	return Assign(tempUstr);
}
CComboError 
CUnicodeString::Assign(const char *str)
{
	ANSI_STRING	ansiString;
	CComboError	error;

	RtlInitAnsiString(&ansiString, str);

	PGPUInt32	newLength	= RtlAnsiStringToUnicodeSize(&ansiString);
	error = Resize(newLength/sizeof(WCHAR));

	if (error.IsntError())
	{
		error.err = RtlAnsiStringToUnicodeString(&mUniData, &ansiString, 
			FALSE);

		if (error.HaveNonPGPError())
			error.pgpErr = kPGPError_StringOpFailed;

		if (error.IsntError())
		{
			mUniData.Length = static_cast<PGPUInt16>(newLength);
			RemoveZeroTermination();
		}
	}

	return error;
}

CComboError 
CUnicodeString::Assign(char c)
{
	WCHAR	wcharStr[2]	= {c, '\0'};
	return Assign(wcharStr);
}

CComboError 
CUnicodeString::Append(const UNICODE_STRING& ustr)
{
	CComboError	error;

	if (mWCharArray.Get() == ustr.Buffer)
		return error;

	PGPUInt32	newLength	= mUniData.Length + ustr.Length;
	error = Resize(newLength/sizeof(WCHAR));

	RemoveZeroTermination();

	if (error.IsntError())
	{
		RtlAppendUnicodeStringToString(&mUniData, 
			const_cast<PUNICODE_STRING>(&ustr));
	}

	return error;
}

CComboError 
CUnicodeString::Append(const WCHAR *wstr)
{
	pgpAssertAddrValid(wstr, WCHAR);

	UNICODE_STRING	tempUstr;
	RtlInitUnicodeString(&tempUstr, wstr);

	return Append(tempUstr);
}

CComboError 
CUnicodeString::Append(const char *str)
{
	pgpAssertStrValid(str);

	CComboError	error;

	CUnicodeString	tempUStr;
	error = tempUStr.Status();

	if (error.IsntError())
		error = tempUStr.Assign(str);

	if (error.IsntError())
		error = Append(tempUStr);

	return error;
}

CComboError 
CUnicodeString::Append(char c)
{
	WCHAR	wcharStr[2]	= {c, '\0'};
	return Append(wcharStr);
}

CComboError 
CUnicodeString::Prepend(const UNICODE_STRING& ustr)
{
	CComboError	error;

	if (mWCharArray.Get() == ustr.Buffer)
		return error;

	PGPUInt32	newLength	= mUniData.Length + ustr.Length;
	error = Resize(newLength/sizeof(WCHAR));

	if (error.IsntError())
	{
		CUnicodeString	tempUStr;

		error = tempUStr.Status();

		if (error.IsntError())
			error = tempUStr.Assign(mUniData);

		if (error.IsntError())
		{
			RtlCopyUnicodeString(&mUniData, 
				const_cast<PUNICODE_STRING>(&ustr));
			RemoveZeroTermination();

			RtlAppendUnicodeStringToString(&mUniData, tempUStr.Get());
		}
	}

	return error;
}

CComboError 
CUnicodeString::Prepend(const WCHAR *wstr)
{
	pgpAssertAddrValid(wstr, WCHAR);

	UNICODE_STRING	tempUstr;
	RtlInitUnicodeString(&tempUstr, wstr);

	return Prepend(tempUstr);
}

CComboError 
CUnicodeString::Prepend(const char *str)
{
	pgpAssertStrValid(str);

	CComboError	error;

	CUnicodeString	tempUStr;
	error = tempUStr.Status();

	if (error.IsntError())
		error = tempUStr.Assign(str);

	if (error.IsntError())
		error = Prepend(tempUStr);

	return error;
}

CComboError 
CUnicodeString::Prepend(char c)
{
	WCHAR	wcharStr[2]	= {c, '\0'};
	return Prepend(wcharStr);
}

CComboError 
CUnicodeString::Resize(PGPUInt32 newSize)
{
	CComboError	error;
	error = mWCharArray.Resize(newSize);

	if (error.IsntError())
	{
		mUniData.Buffer = mWCharArray.Get();
		mUniData.MaximumLength = newSize * sizeof(WCHAR);

		mUniData.Length = pgpMin(mUniData.MaximumLength, mUniData.Length);
	}

	return error;
}

void 
CUnicodeString::Init()
{
	mUniData.Buffer			= mWCharArray.Get();
	mUniData.Length			= 0;
	mUniData.MaximumLength	= mWCharArray.Size();
}
